#define _countof(array) (sizeof(array)/sizeof(array[0]))

// Definitionen der Fixzahlen
const double Epsln					= 1.0e-10;
const double fPi					= 3.141592653589793238;
const double fHalfPi				= fPi * 0.5;
const double fTwoPi					= fPi * 2.0;
const double fEarthCircumference	= 40007000;	// Polar circumference
const double fEarthRadius			= 0.5 * fEarthCircumference / fPi;
const double fDegToRad				= fPi / 180.0;
const double fDegToMeters			= fEarthCircumference / 360.0;
const double fMetersToDeg			= 360.0 / fEarthCircumference;
const double fQuadToLongitude		= 360.0 / 4294967296.0;
const double fMetersToFt			= 3.28084;
const double fFtToMeters			= 0.3048;


inline bool _IfIntersect(double fLatMin0, double fLatMax0, double fLonMin0, double fLonMax0, double fLat1, double fLon1)
{
	return fLonMin0 <= fLon1 && fLonMax0 >= fLon1 && fLatMin0 <= fLat1 && fLatMax0 >= fLat1;
}

inline bool IfIntersect(double fLatMin0, double fLatMax0, double fLonMin0, double fLonMax0, double fLat1, double fLon1)
{
	if (fLonMin0 < fLonMax0)
	{
		return _IfIntersect(fLatMin0, fLatMax0, fLonMin0, fLonMax0, fLat1, fLon1);
	}
	else // if (fLonMin0 >= fLonMax0)
	{
		return _IfIntersect(fLatMin0, fLatMax0, fLonMin0, 180.0, fLat1, fLon1) ||
			   _IfIntersect(fLatMin0, fLatMax0, -180.0, fLonMax0, fLat1, fLon1);
	}
}

inline bool _IfIntersect(double fLatMin0, double fLatMax0, double fLonMin0, double fLonMax0,
						 double fLatMin1, double fLatMax1, double fLonMin1, double fLonMax1)
{
	return fLonMin0 <= fLonMax1 && fLonMax0 >= fLonMin1 && fLatMin0 <= fLatMax1 && fLatMax0 >= fLatMin1;
}

inline bool IfIntersect(double fLatMin0, double fLatMax0, double fLonMin0, double fLonMax0,
						double fLatMin1, double fLatMax1, double fLonMin1, double fLonMax1)
{
	if (fLonMin0 < fLonMax0 && fLonMin1 < fLonMax1)
	{
		return _IfIntersect(fLatMin0, fLatMax0, fLonMin0, fLonMax0, fLatMin1, fLatMax1, fLonMin1, fLonMax1);
	}
	else if (fLonMin0 >= fLonMax0 && fLonMin1 < fLonMax1)
	{
		return _IfIntersect(fLatMin0, fLatMax0, fLonMin0, 180.0, fLatMin1, fLatMax1, fLonMin1, fLonMax1) ||
			   _IfIntersect(fLatMin0, fLatMax0, -180.0, fLonMax0, fLatMin1, fLatMax1, fLonMin1, fLonMax1);
	}
	else if (fLonMin0 < fLonMax0 && fLonMin1 >= fLonMax1)
	{
		return _IfIntersect(fLatMin0, fLatMax0, fLonMin0, fLonMax0, fLatMin1, fLatMax1, fLonMin1, 180.0) ||
			   _IfIntersect(fLatMin0, fLatMax0, fLonMin0, fLonMax0, fLatMin1, fLatMax1, -180.0, fLonMax1);
	}
	else
	{
		return _IfIntersect(fLatMin0, fLatMax0, fLonMin0, 180.0, fLatMin1, fLatMax1, fLonMin1, 180.0) ||
			   _IfIntersect(fLatMin0, fLatMax0, fLonMin0, 180.0, fLatMin1, fLatMax1, -180.0, fLonMax1) ||
			   _IfIntersect(fLatMin0, fLatMax0, -180.0, fLonMax0, fLatMin1, fLatMax1, fLonMin1, 180.0) ||
			   _IfIntersect(fLatMin0, fLatMax0, -180.0, fLonMax0, fLatMin1, fLatMax1, -180.0, fLonMax1);
	}
}

inline DWORD BcdToDWord(DWORD dwBcd)
{
	DWORD dwVal = 0;

	for (int nIndex = 0; nIndex < sizeof(dwBcd) * 2; nIndex++)
	{
		dwVal *= 10;
		dwBcd = _rotl(dwBcd, 4);

		dwVal += dwBcd & 0x0000000f;
	}

	return dwVal;
}

inline DWORD DWordToBcd(DWORD dwVal)
{
	DWORD dwBcd = 0;

	for (int nIndex = 0; nIndex < sizeof(dwBcd) * 2; nIndex++)
	{
		dwBcd += dwVal % 10;
		dwBcd = _rotr(dwBcd, 4);

		dwVal /= 10;
	}

	return dwBcd;
}

inline double asin_safe(double x) { return asin(max(-1, min(x, 1))); }
inline double acos_safe(double x) { return acos(max(-1, min(x, 1))); }
inline double mod(double y, double x) { return y >= 0 ? y - x * int(y / x) : y + x * (int(-y / x) + 1); }

// Distance between points
inline double GlobDist(double fLat1, double fLon1, double fLat2, double fLon2)
{	// fLat1, fLon1 -> from		fLat2, fLon2 -> to
	double cos_Delta_Lon = cos((fLon2 - fLon1) * fDegToRad);
	fLat1 *= fDegToRad;
	fLat2 *= fDegToRad;

//	fEarthRadius * acos(sin(fLat1) * sin(fLat2) + cos(fLat1) * cos(fLat2) * cos_Delta_Lon);
//	sin(fLat1) * sin(fLat2) + cos(fLat1) * cos(fLat2) * cos_Delta_Lon
//	0.5 * cos(fLat2 - fLat1) - 0.5 * cos(fLat2 + fLat1)   +   0.5 * cos(fLat2 - fLat1) * cos_Delta_Lon + 0.5 * cos(fLat2 + fLat1) * cos_Delta_Lon
//	0.5 * cos(fLat2 - fLat1) + 0.5 * cos(fLat2 - fLat1) * cos_Delta_Lon - 0.5 * cos(fLat2 + fLat1) + 0.5 * cos(fLat2 + fLat1) * cos_Delta_Lon
//	0.5 * cos(fLat2 - fLat1) * (1 + cos_Delta_Lon) + 0.5 * cos(fLat2 + fLat1) * (-1 + cos_Delta_Lon)

	double fE = .5 * (cos(fLat2 - fLat1) * (1 + cos_Delta_Lon) - cos(fLat2 + fLat1) * (1 - cos_Delta_Lon));
	return fEarthRadius * acos_safe(fE);
}

// Course between points
inline double GlobHead(double fLat1, double fLon1, double fLat2, double fLon2)
{	// fLat1, fLon1 -> from		fLat2, fLon2 -> to
	double fDeltaLon = (fLon2 - fLon1) * fDegToRad;
	double cos_Delta_Lon = cos(fDeltaLon);
	fLat1 *= fDegToRad;
	fLat2 *= fDegToRad;

//	mod(atan2(sin(fDeltaLon) * cos(fLat2), cos(fLat1) * sin(fLat2) - sin(fLat1) * cos(fLat2) * cos_Delta_Lon), 2 * fPi) / fDegToRad
//	cos(fLat1) * sin(fLat2) - sin(fLat1) * cos(fLat2) * cos_Delta_Lon
//	0.5 * (sin(fLat1 + fLat2) - sin(fLat1 - fLat2)) - 0.5 * (sin(fLat1 + fLat2) + sin(fLat1 - fLat2)) * cos_Delta_Lon
//	0.5 * sin(fLat1 + fLat2) - 0.5 * sin(fLat1 - fLat2) - 0.5 * sin(fLat1 + fLat2) * cos_Delta_Lon - 0.5 * sin(fLat1 - fLat2) * cos_Delta_Lon
//	0.5 * sin(fLat1 + fLat2) - 0.5 * sin(fLat1 + fLat2) * cos_Delta_Lon - 0.5 * sin(fLat1 - fLat2) * cos_Delta_Lon - 0.5 * sin(fLat1 - fLat2)
//	0.5 * sin(fLat1 + fLat2) * (1 - cos_Delta_Lon) - 0.5 * sin(fLat1 - fLat2) * (1 + cos_Delta_Lon)
//	0.5 * (sin(fLat1 + fLat2) * (1 - cos_Delta_Lon) - sin(fLat1 - fLat2) * (1 + cos_Delta_Lon))

	double fE = .5 * (sin(fLat1 + fLat2) * (1 - cos_Delta_Lon) - sin(fLat1 - fLat2) * (1 + cos_Delta_Lon));
	return mod(atan2(sin(fDeltaLon) * cos(fLat2), fE), 2 * fPi) / fDegToRad;
}

// Lat/lon given radial and distance
inline void GlobPos(double& fLat2, double& fLon2, double fLat1, double fLon1, double fHead, double fDist)
{	// fLat1, fLon1 -> from		fLat2, fLon2 -> to
	fLat1 *= fDegToRad;
	fLon1 *= -fDegToRad;
	fHead *= fDegToRad;
	fDist /= fEarthRadius;

	double sin_Lat1 = sin(fLat1);
	double cos_Dist = cos(fDist);
	double fE = sin(fDist) * cos(fLat1);

	fLat2 = asin_safe(cos_Dist * sin_Lat1 + cos(fHead) * fE);
	double fDeltaLon = atan2(sin(fHead) * fE, cos_Dist - sin_Lat1 * sin(fLat2));
	fLon2 = mod(fLon1 - fDeltaLon + fPi, 2 * fPi) - fPi;

	fLat2 /= fDegToRad;
	fLon2 /= -fDegToRad;
}

// Intersecting radials
inline bool GlobPos(double& fLat3, double& fLon3, double fLat1, double fLon1, double fHead1, double fLat2, double fLon2, double fHead2)
{	// from fLat1/fLon1 with fHead1 to fLat3/fLon3 and with fHead2 to fLat2/fLon2
	fLat1 *= fDegToRad;
	fLon1 *= -fDegToRad;
	fHead1 *= fDegToRad;
	fLat2 *= fDegToRad;
	fLon2 *= -fDegToRad;
	fHead2 = mod(fHead2 * fDegToRad + fPi, 2 * fPi);

	double fDst12 = 2 * asin_safe(sqrt(pow(sin((fLat1 - fLat2) / 2), 2) + cos(fLat1) * cos(fLat2) * pow(sin((fLon1 - fLon2) / 2), 2)));

	double fCrs12;
	if (sin(fLon2 - fLon1) < 0)
		fCrs12 = acos_safe((sin(fLat2) - sin(fLat1) * cos(fDst12)) / (sin(fDst12) * cos(fLat1)));
	else
		fCrs12 = 2. * fPi - acos_safe((sin(fLat2) - sin(fLat1) * cos(fDst12)) / (sin(fDst12) * cos(fLat1)));

	double fCrs21;
	if (sin(fLon1 - fLon2) < 0)
		fCrs21 = acos_safe((sin(fLat1) - sin(fLat2) * cos(fDst12)) / (sin(fDst12) * cos(fLat2)));
	else
		fCrs21 = 2. * fPi - acos_safe((sin(fLat1) - sin(fLat2) * cos(fDst12)) / (sin(fDst12) * cos(fLat2)));

	double fAng1 = mod(fHead1 - fCrs12 + fPi, 2. * fPi) - fPi;
	double fAng2 = mod(fCrs21 - fHead2 + fPi, 2. * fPi) - fPi;

	if (sin(fAng1) * sin(fAng2) <= sqrt(Epsln)) 
		return false;

	fAng1 = fabs(fAng1);
	fAng2 = fabs(fAng2);
	double fAng3 = acos_safe(-cos(fAng1) * cos(fAng2) + sin(fAng1) * sin(fAng2) * cos(fDst12));
	double fDst13 = asin_safe(sin(fAng2) * sin(fDst12) / sin(fAng3));
	fLat3 = asin_safe(sin(fLat1) * cos(fDst13) + cos(fLat1) * sin(fDst13) * cos(fHead1));
	fLon3 = mod(fLon1 - asin_safe(sin(fHead1) * sin(fDst13) / cos(fLat3)) + fPi, 2 * fPi) - fPi;

	fLat3 /= fDegToRad;
	fLon3 /= -fDegToRad;
	return true;
}

// Lat/lon between 2 points
inline void GlobPos(double &fLat3, double &fLon3, double fLat1, double fLon1, double fLat2, double fLon2, double fPercent)
{	// fLat1, fLon1 -> from		fLat2, fLon2 -> to		fPercent == 0 -> Lat1/Lon1		fPercent == 1 -> Lat2/Lon2
	if (fPercent > 1)
		fPercent = 1;
	else if (fPercent < 0)
		fPercent = 0;

	if (fabs(fLon1 - fLon2) * fDegToRad <= Epsln || fPercent <= Epsln)
	{
		fLat3 = fLat1 + (fLat2 - fLat1)	* fPercent;
		fLon3 = fLon1;
		return;
	}

	bool bTransform = false;

	if (fabs(fLon1 - fLon2) > 180)
	{	// das kleine Dreieck in den Wertebereich +- 180 Grad transformieren
		if ((fLon1 += 180) > 180)
			fLon1 -= 360;
		if ((fLon2 += 180) > 180)
			fLon2 -= 360;
		bTransform = true;
	}

	double fDeltaLon = (fLon1 - fLon2) * fDegToRad;
	double a = (90 - fLat2) * fDegToRad;
	double b = (90 - fLat1) * fDegToRad;

	double sin_N_Div2 = sin(fDeltaLon / 2);
	double sin_a_Plus_b_Div2 = sin((a + b) / 2);
	double cos_N_Div2 = cos(fDeltaLon / 2);
	double sin_a_Minus_b_Div2 = sin((a - b) / 2);
	double n = 2 * asin(sqrt(sin_N_Div2 * sin_N_Div2 * sin_a_Plus_b_Div2 * sin_a_Plus_b_Div2 + cos_N_Div2 * cos_N_Div2 * sin_a_Minus_b_Div2 * sin_a_Minus_b_Div2));

	double s = (n + a + b) / 2;
	double A = 2 * atan2(sqrt(sin(s - b) * sin(s - n)), sqrt(sin(s) * sin(s - a)));

	double sin_A_Div2 = sin(A / 2);
	double sin_b_Plus_ns_Div2 =	sin((b + n * fPercent) / 2);
	double cos_A_Div2 = cos(A / 2);
	double sin_b_Minus_ns_Div2 = sin((b - n * fPercent) / 2);
	double as = 2 * asin(sqrt(sin_A_Div2 * sin_A_Div2 * sin_b_Plus_ns_Div2 * sin_b_Plus_ns_Div2 + cos_A_Div2 * cos_A_Div2 * sin_b_Minus_ns_Div2 * sin_b_Minus_ns_Div2));

	double ss = (n * fPercent + as + b) / 2;
	double NS = 2 * atan2(sqrt(sin(ss - b) * sin(ss - as)), sqrt(sin(ss) * sin(ss - n * fPercent)));

	fLat3 = 90 - as / fDegToRad;

	if (fLon1 > fLon2)
		fLon3 = fLon1 - NS / fDegToRad;
	else
		fLon3 = fLon1 + NS / fDegToRad;

	if (bTransform)
	{	// die Transformation rckgngig machen
		if ((fLon3 -= 180) < -180)
			fLon3 += 360;
	}
}

inline double DeltaGrad(double fGrad1, double fGrad2)
{
	double fDelta = fabs(fGrad1 - fGrad2);
	return fDelta <= 180. ? fDelta : fabs(360. - fDelta);
}

inline UINT TAStoIAS(double fTAS, double fAlt)	// kts, ft
{
	return (UINT)floor((max(fTAS * (1 - pow(max(fAlt, 0.0) / 100000, 0.865)), 1)) + .5);
}

inline UINT IAStoTAS(double fIAS, double fAlt)	// kts, ft
{
	return (UINT)floor((fIAS / max(1 - pow(max(fAlt, 0.0) / 100000, 0.865), 0.002325581395349)) + .5);
}

inline double TAStoMach(double fTAS, double fTemp)
{
	return fTAS / (38.967854 * sqrt(fTemp + 273.15));
}

inline double FuelMeasure(UINT uMeasureTo, UINT uMeasureFrom, double fVal)
{	// conversion fuel units
	switch (uMeasureFrom)
	{	// to kg
	case FUEL_MEASURE_LBS: fVal = fVal * .45359; break;
	case FUEL_MEASURE_GAL: fVal = fVal * 3.7853 * .815; break;
	case FUEL_MEASURE_KG: fVal = fVal; break;
	case FUEL_MEASURE_L: fVal = fVal * .815; break;
	default: return 0;
	}

	switch (uMeasureTo)
	{
	case FUEL_MEASURE_LBS: fVal = fVal / .45359; break;
	case FUEL_MEASURE_GAL: fVal = fVal / .815 / 3.7853; break;
	case FUEL_MEASURE_KG: fVal = fVal; break;
	case FUEL_MEASURE_L: fVal = fVal / .815; break;
	default: return 0;
	}

	return fVal;
}
