// ScenViewUtil.h: Schnittstelle fr die Klasse CScenViewUtil.
//
//////////////////////////////////////////////////////////////////////

#if !defined(AFX_SCENVIEWUTIL_H__BD6230A1_F861_11D1_A4B6_4854E82880EF__INCLUDED_)
#define AFX_SCENVIEWUTIL_H__BD6230A1_F861_11D1_A4B6_4854E82880EF__INCLUDED_

#if _MSC_VER >= 1000
#pragma once
#endif // _MSC_VER >= 1000

#define	SYMBOL_ZOOM		64

class CZoom
{
public:
	CZoom(int nFaktor = 1, double fCenterLat = 0, double fCenterLon = 0) { m_nFaktor = nFaktor; m_fCenterLat = fCenterLat; m_fCenterLon = fCenterLon; }
	int		m_nFaktor;
	double	m_fCenterLat;	// Top = 90, Bottom = -90
	double	m_fCenterLon;	// Left = -180, Right = 180
};

#define CArray<CZoom, CZoom&>	CArrayZoom;

class CScenViewUtil  
{
public:
	CScenViewUtil();
	virtual ~CScenViewUtil();
	void	SetRoundWorld(bool bSet) { m_bRoundWorld = bSet; }
	bool	GetRoundWorld() { return m_bRoundWorld;	}
	int		SetZoom(int nFaktor, double fCenterLat, double fCenterLon);
	int		GetZoom();
	int		ZoomOut();
	void	SetCenter(double fCenterLat, double fCenterLon, bool bAll = false);
	double	GetCenterLon() { return m_ZoomArray.GetAt(m_ZoomArray.GetUpperBound()).m_fCenterLon; }
	double	GetCenterLat() { return m_ZoomArray.GetAt(m_ZoomArray.GetUpperBound()).m_fCenterLat; }
	void	GetViewArea(int nViewWidth, int nViewHeight, double& fLatMin, double& fLatMax, double& fLonMin, double& fLonMax);

	bool	GlobToScreen(long &lX, long &lY, double fLat, double fLon, int nClientWidth, int nClientHeight, bool fInside = false);
	bool	ScreenToGlob(double &fLat, double &fLon, long lX, long lY, int nClientWidth, int nClientHeight);
	int		DistToPixels(int nClientWidth, int nClientHeight, double fDist);
	double	PixelsToDist(int nClientWidth, int nClientHeight, int nPixels);
private:
	void	CalculateCenter();
	CArrayZoom	m_ZoomArray;
	bool		m_bRoundWorld;
	double		m_fSinLatCenter, m_fCosLatCenter, m_fSinLonCenter, m_fCosLonCenter;
};

//////////////////////////////////////////////////////////////////////
// public

// transformation from Lat/Lon to screen coordinates
inline bool CScenViewUtil::GlobToScreen(long &lX, long &lY, double fLat, double fLon, int nClientWidth, int nClientHeight, bool fInside)
{
	bool bResult = true;
	int nZoom = GetZoom();

	if (!GetRoundWorld())
	{
		double	fXT = (fLon - GetCenterLon()) / 180.0;
		double	fYT = (fLat - GetCenterLat()) / 90.0;

		if (fXT < -1.)
			fXT += 2.;
		else if (fXT > 1.)
			fXT -= 2.;

		UINT uTmp = nZoom * max(nClientWidth / 2, nClientHeight);
		lX =  long(uTmp * fXT);
		lY = -long(uTmp / 2 * fYT);
	}
	else
	{
		fLat *= fDegToRad;
		double	fSinLat = sin(fLat);
		double	fCosLat = cos(fLat);
		double	fLonT = (fLon - GetCenterLon()) * fDegToRad;
		double	fCosLat_CosLonDelta = cos(fLonT) * fCosLat;
		double	fXT = sin(fLonT) * fCosLat;
		double	fYT = m_fSinLatCenter * fCosLat_CosLonDelta + m_fCosLatCenter * fSinLat;
		bResult = m_fCosLatCenter * fCosLat_CosLonDelta - m_fSinLatCenter * fSinLat >= 0.;

		double fAbs = asin(sqrt(fXT * fXT + fYT * fYT)) / fHalfPi;
		double fGrad = atan2(fXT, fYT);
		fXT = fAbs * sin(fGrad);
		fYT = fAbs * cos(fGrad);

		int nTmp = nZoom * min(nClientWidth, nClientHeight) / 2;
		lX =  long(nTmp * fXT);
		lY = -long(nTmp * fYT);
	}

	lX += nClientWidth / 2;
	lY += nClientHeight / 2;

	long lXDelta, lYDelta;

	if (fInside)
	{
		lXDelta = nClientWidth / 10;
		lYDelta = nClientHeight / 10;
	}
	else
	{
		lXDelta = -nClientWidth / 5;
		lYDelta = -nClientHeight / 5;
	}

	return bResult && lX >= lXDelta && lX <= nClientWidth - lXDelta && lY >= lYDelta && lY <= nClientHeight - lYDelta;
}

// transformation from screen coordinates to Lat/Lon
inline bool CScenViewUtil::ScreenToGlob(double &fLat, double &fLon, long lX, long lY, int nClientWidth, int nClientHeight)
{
	double fX = lX - nClientWidth * .5;
	double fY = lY - nClientHeight * .5;
	int nZoom = GetZoom();

	if (!GetRoundWorld())
	{
		double fXT =  fX      / (nZoom * max(nClientWidth * .5, nClientHeight));
		double fYT = -fY * 2. / (nZoom * max(nClientWidth * .5, nClientHeight));

		fLon = fXT * 179.999999 + GetCenterLon();
		fLat = fYT *  89.999999 + GetCenterLat();

		if (fLon < -180.)
			fLon += 360.;
		else if (fLon > 180.)
			fLon -= 360.;

		if (fLat < -90. || fLat > 90.)
			return false;
	}
	else
	{
		double fXT =  fX * 2. / (nZoom * min(nClientWidth, nClientHeight));
		double fYT = -fY * 2. / (nZoom * min(nClientWidth, nClientHeight));

		if (fXT * fXT + fYT * fYT > 1.)
			return false;

		double fAbs = sin(sqrt(fXT * fXT + fYT * fYT) * fHalfPi);
		double fGrad = atan2(fXT, fYT);
		fXT = fAbs * sin(fGrad);
		fYT = fAbs * cos(fGrad);

		double fZT = sqrt(1. - (fXT * fXT + fYT * fYT));
	
		double	fY_SinLatCenter = fYT * m_fSinLatCenter;
		double	fZ_CosLatCenter = fZT * m_fCosLatCenter;

		fLon = atan2(fXT * m_fCosLonCenter - m_fSinLonCenter * (fY_SinLatCenter + fZ_CosLatCenter),
					 fXT * m_fSinLonCenter + m_fCosLonCenter * (fY_SinLatCenter + fZ_CosLatCenter)) / fDegToRad;
									   															   
		fLat = asin(fYT * m_fCosLatCenter - fZT * m_fSinLatCenter) / fDegToRad;
	}

	return true; 
}

inline int CScenViewUtil::DistToPixels(int nClientWidth, int nClientHeight, double fDist)
{
	if (!GetRoundWorld())
		return int(fDist / fEarthCircumference * max(nClientWidth, nClientHeight * 2) * GetZoom());
	else
		return int(fDist / (fEarthCircumference * .5) * min(nClientWidth, nClientHeight) * GetZoom());
}

inline double CScenViewUtil::PixelsToDist(int nClientWidth, int nClientHeight, int nPixels)
{
	if (!GetRoundWorld())
		return fEarthCircumference * nPixels / max(nClientWidth, nClientHeight * 2) / GetZoom();
	else
		return (fEarthCircumference * .5) * nPixels / min(nClientWidth, nClientHeight) / GetZoom();
}

#endif // !defined(AFX_SCENVIEWUTIL_H__BD6230A1_F861_11D1_A4B6_4854E82880EF__INCLUDED_)
